<?php

namespace App\Http\Controllers;

use App\CatalogoCampo;
use Illuminate\Http\Request;
use App\OrdenProduccion;
use App\OrdenProduccionParcial;
use App\IngresoAlmacen;
use App\SalidaAlmacen;
use App\Cotizacion;
use App\CotizacionProductos;
use App\Cliente;
use App\Inventario;
use App\InventarioDetalle;
use App\Producto;
use App\InventarioMateriaPrimas;
use App\InventarioMateriaPrimasSalida;
use App\ProductoMedidas;
use PDF;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class InventarioController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
        $opParciales = OrdenProduccionParcial::where('status', 1)->with('op')->get();
        return view('inventario.dashboard', compact('opParciales'));
    }

    // INSUMOS Y ACCESORIOS

    public function dashboardInsumos()
    {
        $inventarioInsumos = Inventario::all();
        return view('inventario.dashboard_insumo', compact('inventarioInsumos'));
    }

    public function ingresoInsumo()
    {
        $medidas = ProductoMedidas::orderBy('nombre', 'asc')->get();
        $tallas = CatalogoCampo::where('tipo_campo', 2)->orderBy('nombre', 'asc')->get();
        $colores = CatalogoCampo::where('tipo_campo', 1)->orderBy('nombre', 'asc')->get();
        $clientes = Cliente::orderBy('nombre', 'asc')->get();
        return view('inventario.nuevo_insumo', compact('clientes', 'medidas', 'tallas', 'colores'));
    }

    public function guardarInsumo(Request $request)
    {
        // Capturar los datos del formulario
        $data = $request->all();

        // Inicializar el arreglo detalle
        $detalle = [];

        // Variable para verificar si hay algún faltante
        $hayFaltante = false;

        // Recorrer las descripciones y reestructurar los datos
        foreach ($data['descripcion'] as $index => $descripcion) {
            $faltante = $data['faltante'][$index] ?? null;
            if ($faltante > 0) {
                $hayFaltante = true;
            }

            $detalle[] = [
                'descripcion' => $descripcion,
                'observaciones' => $data['observaciones'][$index] ?? null,
                'unidad_medida_id' => $data['medida'][$index] ?? null,
                'estilo' => $data['estilo'][$index] ?? null,
                'talla_id' => $data['talla'][$index] ?? null,
                'color_id' => $data['color'][$index] ?? null,
                'cantidad_esperada' => $data['cantidad_esperada'][$index] ?? null,
                'cantidad_recibida' => $data['cantidad_ingresada'][$index] ?? null,
                'faltante' => $faltante,
                'cajas' => $data['cajas'][$index] ?? null,
                'estado' => $faltante > 0 ? 2 : 1,
            ];
        }

        // Manejar el almacenamiento de la imagen
        $imagenNombre = null;
        if ($request->hasFile('imagen')) {
            $imagen = $request->file('imagen');
            $hash = md5($imagen->getClientOriginalName() . microtime());
            $extension = $imagen->getClientOriginalExtension();
            $filename = $hash . '.' . $extension;
            $imagen->storeAs('public/inventarios', $filename);
            $imagenNombre = $filename;
        }

        // Crear la nueva estructura
        $newData = [
            'cliente_id' => $data['cliente'] ?? null,
            'linea_negocio_id' => $data['linea_negocio'] ?? null,
            'tipo_inventario_id' => 1,
            'po' => $data['po'] ?? null,
            'descripcion' => $data['descripcion_general'] ?? null,
            'observaciones' => $data['observaciones_general'] ?? null,
            'estado' => $hayFaltante ? 2 : 1, // Cambiar el estado según la variable de control
            'fecha_ingreso' => $data['fecha_ingreso'] ?? null,
            'imagen' => $imagenNombre,
            'reclamo' => $hayFaltante ? 1 : 0,
        ];

        $newInventario = Inventario::create($newData);
        $newInventario->detalle()->createMany($detalle);

        $arrResponse = [
            'message' => 'Se ha creado el inventario correctamente'
        ];

        return redirect()->route('inventario.dashboard_insumo')->with($arrResponse);
    }



    public function detalleInsumo($insumo_id)
    {
        $medidas = ProductoMedidas::orderBy('nombre', 'asc')->get();
        $tallas = CatalogoCampo::where('tipo_campo', 2)->orderBy('nombre', 'asc')->get();
        $colores = CatalogoCampo::where('tipo_campo', 1)->orderBy('nombre', 'asc')->get();
        $insumo = Inventario::where('id', $insumo_id)->with('detalle')->first();
        $clientes = Cliente::orderBy('nombre', 'asc')->get();
        return view('inventario.detalle_insumo', compact('clientes', 'insumo', 'medidas', 'tallas', 'colores'));
    }

    public function generarReclamoInsumo($insumo_id)
    {
        $insumo = Inventario::where('id', $insumo_id)->first();
        $insumoFaltante = $insumo->detalle->where('faltante', '>', 0);
        $pdf = PDF::loadView('inventario.pdf.reclamo', compact('insumo', 'insumoFaltante'));
        $fileName = 'reclamo_insumo_' . $insumo->id . '_' . date('Ymd') . '.pdf';
        return $pdf->download($fileName);
    }

    public function detalleInsumoApiPublic($insumo_id)
    {
        try {
            $insumo = Inventario::with('detalle')->findOrFail($insumo_id);
            return response()->json($insumo);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Insumo not found'], 404);
        }
    }

    // ESTADOS INSUMO
    // 1 => 'Ingreso Completo',
    // 2 => 'Ingreso Incompleto',
    // 3 => 'Salida Parcial',
    // 4 => 'Salida Completa',
    // 99 => 'Solicitud de Anulación',
    // 999 => 'Anulado',

    public function entregaParcialInsumo(Request $request)
    {
        // Capturar los datos del formulario
        $data = $request->all();

        // Validar los datos (opcional, pero recomendado)
        $request->validate([
            'idsDetalle' => 'required|array',
            'cantidad_entregar' => 'required|array',
            'insumo_id' => 'required|exists:inventarios,id',
        ]);

        $pendienteEntrega = true;
        // Iterar sobre cada detalle y actualizar la cantidad entregada
        foreach ($data['idsDetalle'] as $index => $detalleId) {
            $cantidadEntregar = $data['cantidad_entregar'][$index];

            // Encontrar el detalle por su ID
            $detalle = InventarioDetalle::find($detalleId);

            if ($detalle) {
                // Actualizar la cantidad entregada
                $detalle->entregado = $cantidadEntregar;
                $detalle->save();

                // Verificar si hay pendiente de entrega
                if ($detalle->entregado < $detalle->cantidad_recibida && $pendienteEntrega) {
                    $pendienteEntrega = true;
                } else {
                    $pendienteEntrega = false;
                }
            }
        }

        $insumo = Inventario::find($data['insumo_id']);
        $insumo->estado = $pendienteEntrega ? 3 : 4;
        $insumo->save();

        $arrayResponse = [
            'message' => 'Se ha actualizado la entrega parcial correctamente'
        ];

        return redirect()->route('inventario.dashboard_insumo')->with($arrayResponse);
    }


    public function actualizarInsumo(Request $request, $insumo_id)
    {
        $insumoActualizar = Inventario::where('id', $insumo_id)->firstOrFail();

        // Capturar los datos del formulario
        $data = $request->all();

        // Inicializar el arreglo detalle
        $detalle = [];

        // Variable para verificar si hay algún faltante
        $hayFaltante = false;

        // Recorrer las descripciones y reestructurar los datos
        foreach ($data['descripcion'] as $index => $descripcion) {
            $faltante = $data['faltante'][$index] ?? 0;
            if ($faltante > 0) {
                $hayFaltante = true;
            }

            $detalle[] = [
                'descripcion' => $descripcion,
                'observaciones' => $data['observaciones'][$index] ?? null,
                'unidad_medida_id' => $data['medida'][$index] ?? null,
                'estilo' => $data['estilo'][$index] ?? null,
                'talla_id' => $data['talla'][$index] ?? null,
                'color_id' => $data['color'][$index] ?? null,
                'cantidad_esperada' => $data['cantidad_esperada'][$index] ?? null,
                'cantidad_recibida' => $data['cantidad_ingresada'][$index] ?? null,
                'faltante' => $faltante,
                'cajas' => $data['cajas'][$index] ?? null,
                'estado' => $faltante > 0 ? 2 : 1,
            ];
        }

        // Manejar el almacenamiento de la imagen
        $imagenNombre = $insumoActualizar->imagen; // Conservar la imagen existente si no se carga una nueva
        if ($request->hasFile('imagen')) {
            $imagen = $request->file('imagen');
            $hash = md5($imagen->getClientOriginalName() . microtime());
            $extension = $imagen->getClientOriginalExtension();
            $filename = $hash . '.' . $extension;
            $imagen->storeAs('public/inventarios', $filename);
            $imagenNombre = $filename;
        }

        // Crear la nueva estructura
        $newData = [
            'cliente_id' => $data['cliente'] ?? null,
            'linea_negocio_id' => $data['linea_negocio'] ?? null,
            'tipo_inventario_id' => 1,
            'po' => $data['po'] ?? null,
            'descripcion' => $data['descripcion_general'] ?? null,
            'observaciones' => $data['observaciones_general'] ?? null,
            'estado' => $hayFaltante && $insumoActualizar->estado < 3 ? 2 : 1, // Cambiar el estado según la variable de control
            'fecha_ingreso' => $data['fecha_ingreso'] ? date('Y-m-d', strtotime($data['fecha_ingreso'])) : null,
            'reclamo' => $hayFaltante ? 1 : 0,
            'imagen' => $imagenNombre,
        ];

        // Actualizar los datos del insumo
        $insumoActualizar->update($newData);

        // Eliminar los detalles antiguos
        $insumoActualizar->detalle()->delete();

        // Crear nuevos detalles
        $insumoActualizar->detalle()->createMany($detalle);

        $arrayResponse = [
            'message' => 'Se ha actualizado el inventario correctamente'
        ];

        return redirect()->route('inventario.dashboard_insumo')->with($arrayResponse);
    }


    // INSUMOS Y ACCESORIOS

    public function ordenProduccionRecibido(OrdenProduccionParcial $ordenProduccionP)
    {
        $opParciales = OrdenProduccionParcial::where('id', $ordenProduccionP->id)->first();
        $op = OrdenProduccion::where('id', $ordenProduccionP->id_op)->with('cliente')->with('producto')->first();
        return view('inventario.recibir_orden_produccion', compact('op', 'opParciales'));
    }
    public function salidaAlamacen()
    {
        $salidasAlmacen = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', '!=', 999999)
            ->where('id_op', '!=', 9999999)
            ->with('cliente')
            ->with('producto')
            ->with('op')
            ->groupBy('id_op')
            ->selectRaw('*,sum(cantidad) as total_entrega')
            ->get();
        $salidasAlmacenInventario = IngresoAlmacen::where('status', 1)
            ->where('id_cotizacion_producto', '>', 0)
            ->with('cliente')
            ->with('producto')
            ->get();


        //dd($salidasAlmacenInventario);
        return view('inventario.dashboard_salida', compact('salidasAlmacen', 'salidasAlmacenInventario'));
    }


    public function HistorialDespacho()
    {
        $salidasAlmacen = IngresoAlmacen::where('id_cliente', '!=', 999999)
            ->where('id_op', '!=', 9999999)
            ->with('cliente')
            ->with('producto')
            ->with('op')
            ->groupBy('id_op')
            ->selectRaw('*,sum(cantidad) as total_entrega')
            ->get();
        $salidasAlmacenInventario = IngresoAlmacen::where('id_cotizacion_producto', '>', 0)
            ->with('cliente')
            ->with('producto')
            ->get();


        //dd($salidasAlmacenInventario);
        //dd($salidasAlmacen);
        return view('inventario.historial_despacho', compact('salidasAlmacen', 'salidasAlmacenInventario'));
    }

    public function archivarSalida(Request $request)
    {

        $salida = IngresoAlmacen::where('id', $request->id)->first();
        $salida->observaciones =  $request->mensaje;
        $salida->status =  77;
        $salida->update();

        $ordenes = OrdenProduccionParcial::where('id_op', $salida->id_op)->get();
        if ($ordenes) {
            foreach ($ordenes as $orden) {
                $orden->estatus =  77;
                $orden->obseervacion_almacen = $request->mensaje;
                $orden->update();
            }
            $cotizacion_producto = CotizacionProductos::where('id', $salida->id_cotizacion_producto)->first();
            $cotizacion_producto->estatus = 77;
            $cotizacion_producto->update();
        }


        $arrResponse = [
            'message' => 'Se archivo correctamente la cotización y salida de almacen'
        ];
        return redirect()->route('inventario.dashboard_salida')->with($arrResponse);
    }

    public function reverseEnvio(Request $request)
    {
        DB::beginTransaction();
        try {
            $salida = SalidaAlmacen::find($request->envio_id);
            $op = OrdenProduccion::where('id', $salida->id_op)->first();
            $op->cantidad_entregada = $op->cantidad_entregada - $salida->cantidad;
            $op->status =  3;
            $op->update();

            $salida->delete();

            $arrResponse = [
                'message' => 'Se revirtio correctamente la salida de almacen'
            ];

            DB::commit();

            return response($arrResponse);
        } catch (\Throwable $th) {
            DB::rollBack();
            $arrResponse = [
                'message' => 'Error al revertir la salida de almacen: ' . $th->getMessage()
            ];
            return response($arrResponse, 500);
        }
    }

    public function dashboardInventario()
    {
        $inventario = InventarioDetalle::whereColumn('cantidad_recibida', '>', 'entregado')
            ->get();
        return view('inventario.dashboard_inventario_insumos', compact('inventario'));
    }

    public function dashboardInventarioMateriaPrima()
    {
        $inventario = InventarioMateriaPrimas::where('total', '>', 0)
            ->select('id_producto', 'created_at', DB::raw('SUM(total) as total'))
            ->groupBy('id_producto')
            ->get();
        return view('inventario.dashboard_inventario_prima', compact('inventario'));
    }
    public function dashboardInventarioMateriaPrimaSalidas()
    {
        $inventario = InventarioMateriaPrimasSalida::all();
        return view('inventario.dashboard_inventario_prima_salida', compact('inventario'));
    }
    public function salidaAlmacenFactura(Request $request)
    {
        $totalProduccion = $request->cantidad - $request->produccion;
        $x = new SalidaAlmacen();
        $x->id_usuario =  Auth::id();
        $x->id_op = $request->id_op;
        $x->id_cliente = $request->id_cliente;
        $x->id_producto = $request->id_producto;
        $x->cantidad = $request->cantidad;
        $x->estatus = 1;
        $x->observaciones = $request->observaciones ?? '';
        $x->save();

        $op = OrdenProduccion::where('id', $request->id_op)->first();
        $cantidadTotalEntregada = $op->cantidad_entregada + $request->cantidad;
        $op->cantidad_entregada = $cantidadTotalEntregada;
        $op->update();
        //Quitar de inventario

        while ($totalProduccion > 0) {
            $inventario = IngresoAlmacen::where('status', 1)
                ->where('id_cliente', 999999)
                ->where('cantidad', '>', 0)
                ->where('id_producto', $request->id_producto)
                ->first();

            if ($inventario->cantidad >= $totalProduccion) {
                $inventario->cantidad = $inventario->cantidad - $totalProduccion;
                $inventario->update();

                $op = OrdenProduccion::where('id', $request->id_op)->first();
                $op->cantidad_inventario = $op->cantidad_inventario + $totalProduccion;
                $op->update();
                $totalProduccion = -1;
            } else {
                $totalProduccion = $totalProduccion - $inventario->cantidad;
                $inventario->status = 2;
                $inventario->cantidad = 0;
                $inventario->update();

                $op = OrdenProduccion::where('id', $request->id_op)->first();
                $op->cantidad_inventario = $op->cantidad_inventario + $inventario->cantidad;
                $op->update();
            }
        }
        $arrResponse = [
            'message' => 'Salida de almacen correctamente'
        ];
        return response($x);
    }
    public function salidaAlmacenFacturacion(Request $request)
    {
        $op = SalidaAlmacen::where('id', $request->id_salida)->with('cliente')->first();
        $cantidad_cotizacion = $op->cotizacion->cantidad ?? 0;
        $cantidad_op = $op->op->cotizaciones_p->cantidad ?? 0;
        $cantidad = $op->cantidad ?? 0;
        if ($cantidad_cotizacion <= $cantidad && $cantidad_cotizacion != 0) {
            $op->estatus = 6;
        } elseif ($cantidad_op <= $cantidad && $cantidad_op != 0) {
            $op->estatus = 6;
        } else {
            $op->estatus = 2;
        }
        $op->update();
        $id_salida = $request->id_salida;
        $no_cotizacion = $op->id_cotizacion > 0 ? $op->id_cotizacion : $op->op->id_cotizacion_producto;
        $info = [
            'idSalida' => $id_salida,
            'idCotizacion' => $no_cotizacion,
            'nombreCliente' => $op->cliente->referencia
        ];

        Mail::send('email.envio_pendiente_factura', compact('info'), function ($message) use ($id_salida) {
            $message->from('info@laravelrp.com', 'Concisa - Despacho realizado listo para facturar ' . $id_salida);
            //$message->to('kicas48@gmail.com');
            $message->to('correo@gmail.com');
            //$message->to('esmeralda@concisa.com.gt');
            $message->subject('Despacho realizado listo para facturar - ' . $id_salida);
        });
        return response($op);
    }
    public function enviosAlamacen()
    {
        $salidas = SalidaAlmacen::where('estatus', 1)
            ->where('id_cliente', '>', 0)
            ->where('id_producto', '>', 0)
            ->with('cliente')
            ->with('producto')
            ->get();
        //dd($salidas);
        return view('inventario.dashboard_envios', compact('salidas'));
    }
    public function generarEnvio($idEnvio)
    {

        $salidas = SalidaAlmacen::where('id', $idEnvio)
            ->with('cliente')
            ->with('producto')
            ->first();
        //dd($salidas);
        $info_envio = [
            'id' => $salidas->id,
            'referencia' => $salidas->cliente->referencia,
            'dia' => date("d"),
            'mes' => date("m"),
            'ano' => date("Y"),
            'cliente' => $salidas->cliente->nombre,
            'cantidad' => $salidas->cantidad,
            'medida' => $salidas->producto->medida->nombre,
            'producto' => $salidas->producto->nombre,
        ];
        $pdf = PDF::loadView('inventario.generar_envio', compact('info_envio'));
        return $pdf->download('envio - ' . $salidas->id . '.pdf');
        dd();
    }

    public function salidaAlmacenFacturaInventario(Request $request)
    {
        $totalProduccion = $request->cantidad;
        $noActualizar = 1;
        while ($totalProduccion > 0) {
            $inventario = IngresoAlmacen::where('status', 1)
                ->where('id_cliente', 999999)
                ->where('cantidad', '>', 0)
                ->where('id_producto', $request->id_producto)
                ->first();
            if ($inventario->cantidad >= $totalProduccion) {
                $inventario->cantidad = $inventario->cantidad - $totalProduccion;
                $inventario->update();
                if ($noActualizar == 1) {
                    $x = new SalidaAlmacen();
                    $x->id_usuario =  Auth::id();
                    $x->id_op = 9999999;
                    $x->id_cliente =  $request->id_cliente;
                    $x->id_producto =  $request->id_producto;
                    $x->cantidad = $totalProduccion;
                    $x->estatus = 1;
                    $x->observaciones = $request->observaciones ?? '';
                    $x->save();
                }
                $totalProduccion = -1;
            } else {
                $totalProduccion = $totalProduccion - $inventario->cantidad;
                $inventario->status = 2;
                $inventario->cantidad = 0;
                $inventario->update();
                if ($noActualizar == 1) {
                    $x = new SalidaAlmacen();
                    $x->id_usuario =  Auth::id();
                    $x->id_op = 9999999;
                    $x->estatus = 1;
                    $x->id_producto =  $request->id_producto;
                    $x->id_cliente =  $request->id_cliente;
                    $x->cantidad = $request->cantidad;
                    $x->observaciones = $request->observaciones ?? '';
                    $x->save();
                }


                $noActualizar = 2;
            }
        }


        return response($x);
    }



    public function salidaAlmacenFacturaInventarioEliminar(Request $request)
    {
        $totalProduccion = $request->cantidad;
        $inventario = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', 999999)
            ->where('cantidad', '>', 0)
            ->where('id_producto', $request->id_producto)
            ->first();
        $noActualizar = 1;

        while ($totalProduccion > 0) {
            if ($inventario->cantidad >= $totalProduccion) {
                $inventario->cantidad = $inventario->cantidad - $totalProduccion;
                $inventario->update();
                if ($noActualizar == 1) {
                    $x = new SalidaAlmacen();
                    $x->id_usuario =  Auth::id();
                    $x->id_op = 9999999;
                    $x->id_cliente =  9999999;
                    $x->cantidad = $totalProduccion;
                    $x->estatus = 3; //Desperdicio
                    $x->id_producto = $request->id_producto;
                    $x->observaciones = $request->observaciones ?? '';
                    $x->save();
                }
                $totalProduccion = -1;
            } else {
                $totalProduccion = $totalProduccion - $inventario->cantidad;
                $inventario->status = 2;
                $inventario->cantidad = 0;
                $inventario->update();
                if ($noActualizar == 1) {
                    $x = new SalidaAlmacen();
                    $x->id_usuario =  Auth::id();
                    $x->id_op = 9999999;
                    $x->estatus = 3; //Desperdicio
                    $x->id_cliente =  9999999;
                    $x->cantidad = $request->cantidad;
                    $x->id_producto = $request->id_producto;
                    $x->observaciones = $request->observaciones ?? '';
                    $x->save();
                }
                $noActualizar = 2;
            }
        }
        $producto = Producto::where('id', $request->id_producto)->with('medida')->first();
        $info = [
            'producto' => $producto->nombre,
            'cantidad' => $request->cantidad,
            'observacion' => $request->observaciones ?? '',
            'medida' => $producto->medida->nombre
        ];
        $idOP = $inventario->id;
        Mail::send('email.inventario_eliminar', compact('info'), function ($message) use ($idOP) {
            $message->from('info@laravelrp.com', 'Concisa - Desechar Producto ' . $idOP);
            $message->to('kicas48@gmail.com');
            $message->to('anylopeza3@gmail.com');
            $message->to('erickfernando85@gmail.com');
            $message->subject('Desechar producto - ' . $idOP);
        });
        return response($x);
    }
    public function salidaAlmacenInventario(Request $request)
    {
        //dd('entro');
        $salidaAlmacen = IngresoAlmacen::where('id', $request->id_inventario)->first();
        $cantidad_solicitud = floatval($request->cantidad);
        $cantidad_orden = floatval($salidaAlmacen->cantidad);
        if ($cantidad_solicitud == $cantidad_orden) {
            $salidaAlmacen->id_cliente = 999999;
            $salidaAlmacen->update();
        } else {
            $op = new IngresoAlmacen();
            $op->id_usuario =  Auth::id();
            $op->id_cliente = 999999;
            $op->id_producto = $request->id_producto;
            $op->cantidad = $cantidad_solicitud;
            $op->id_op = $request->id_op;
            $op->id_op_parcial = 0;
            $op->observaciones = $request->observaciones ?? '';
            $op->status = 1;
            $op->save();

            $salidaAlmacen->cantidad = $cantidad_orden - $cantidad_solicitud;
            $salidaAlmacen->update();
        }
        return response(200);
    }
    public function salidaAlmacenTerminar(Request $request)
    {
        if ($request->cantidad > 0) {

            if ($request->cantidad <= $request->produccion) {
                echo 'Entro1';
                $x = new SalidaAlmacen();
                $x->id_usuario =  Auth::id();
                $x->id_op = $request->id_op;
                $x->id_cliente = $request->id_cliente;
                $x->cantidad = $request->cantidad;
                $x->id_producto = $request->id_producto;
                $x->estatus = 1;
                $x->observaciones = $request->observaciones ?? '';
                $x->save();
            }
            $op = OrdenProduccion::where('id', $request->id_op)->first();
            $op->cantidad_entregada = $op->cantidad_entregada + $request->cantidad;
            $op->status = 5;
            $op->update();
            echo 'Entro2';
        } else {
            echo 'Entro3';
            $op = OrdenProduccion::where('id', $request->id_op)->first();
            $op->status = 5;
            $op->update();
        }
        $arrResponse = [
            'message' => 'Salida de almacen correctamente'
        ];
        return response(200);
    }
    public function salidaAlmacenInventarioTerminar(Request $request)
    {
        $cantidad = floatval($request->cantidad);
        $inventario = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', 999999)
            ->where('cantidad', '>', 0)
            ->where('id_producto', $request->id_producto)
            ->get();
        foreach ($inventario as $in) {
            if ($cantidad > 0) {
                $cantidad_inventario = floatval($in->cantidad);
                if ($cantidad <= $cantidad_inventario) {
                    $resultado = $cantidad_inventario - $cantidad;
                    $cantidad = 0;
                    $inventarioActualizar = IngresoAlmacen::where('id', $in->id)->first();
                    $inventarioActualizar->cantidad = $resultado;
                    $inventarioActualizar->update();
                } else {
                    $cantidad = $cantidad - $cantidad_inventario;
                    $inventarioActualizar = IngresoAlmacen::where('id', $in->id)->first();
                    $inventarioActualizar->cantidad = 0;
                    $inventarioActualizar->update();
                }
            }
        }
        if ($cantidad == 0) {
            $ingresoActualizar = IngresoAlmacen::where('id', $request->id_salida)->first();
            $ingresoActualizar->status = 2;
            $ingresoActualizar->update();

            $x = new SalidaAlmacen();
            $x->id_usuario =  Auth::id();
            $x->id_cotizacion = $request->id_cotizacion;
            $x->cantidad = $request->cantidad;
            $x->id_producto = $request->id_producto;
            $x->id_cliente = $ingresoActualizar->id_cliente;
            $x->estatus = 1;
            $x->observaciones = $request->observaciones ?? '';
            $x->save();
            $arrResponse = [
                'message' => 'Salida de almacen correctamente'
            ];
        } else {
            $arrResponse = [
                'message' => 'No se ingreso la cantidad'
            ];
        }

        return response(200);
    }
    public function historialProducto($idProducto)
    {
        $productos = IngresoAlmacen::where('id_producto', $idProducto)->where('status', 1)->with('producto')->get();
        return view('inventario.dashboard_producto', compact('productos'));
    }
    public function ingresoAlmacen(Request $request)
    {
        $op = new IngresoAlmacen();
        $op->id_usuario =  Auth::id();
        $op->id_cliente = $request->id_cliente;
        $op->id_producto = $request->id_producto;
        $op->cantidad = $request->cantidad;
        $op->id_op = $request->id_op;
        $op->id_op_parcial = $request->id_op_parcial;
        $op->fecha_recibida = date("Y-m-d", strtotime($request->fecha_recibida));
        $op->id_cotizacion = $request->id_cotizacion;
        $op->observaciones = $request->observaciones ?? '';
        $op->status = 1;
        $op->save();

        $op = OrdenProduccionParcial::where('id', $request->id_op_parcial)->first();
        $op->status = 99;
        $op->update();

        $arrResponse = [
            'message' => 'Ingreso almacen correctamente'
        ];
        return redirect()->route('inventario.dashboard')->with($arrResponse);
    }
    public function ingresoAlmacenAjax(Request $request)
    {
        $opParciales = OrdenProduccionParcial::where('id', $request->idOrden)->with('op')->first();
        $ingreso_almacen = new IngresoAlmacen();
        $ingreso_almacen->id_usuario =  Auth::id();
        $ingreso_almacen->id_cliente = $opParciales->op->id_cliente;
        $ingreso_almacen->id_producto = $opParciales->id_producto;
        $ingreso_almacen->cantidad = $opParciales->cantidad;
        $ingreso_almacen->id_op = $opParciales->op->id;
        $ingreso_almacen->id_op_parcial = $opParciales->id;
        $ingreso_almacen->observaciones = $opParciales->op->observaciones ?? '';
        $ingreso_almacen->status = 1;
        $ingreso_almacen->save();

        $opParciales->status = 99;
        $opParciales->update();

        $arrResponse = [
            'message' => 'Ingreso almacen correctamente'
        ];
        return response($ingreso_almacen);
    }
    public function dashboardArchivo()
    {
        //
        $salidasAlmacen = IngresoAlmacen::where('status', 1)
            ->where('id_cliente', '!=', 999999)
            ->with('cliente')
            ->with('producto')
            ->with('op')
            ->groupBy('id_op')
            ->selectRaw('*,sum(cantidad) as total_entrega')
            ->get();
        $anuladasAlmacen = IngresoAlmacen::where('status', 77)
            ->where('id_cliente', '!=', 999999)
            ->with('cliente')
            ->with('producto')
            ->get();

        return view('inventario.dashboard_archivo', compact('salidasAlmacen', 'anuladasAlmacen'));
    }
    public function ajusteInventario()
    {

        return view('inventario.ajuste_inventario');
    }
    public function ajusteInventarioPrima()
    {
        return view('inventario.ajuste_inventario_prima');
    }
    public function ajusteInventarioGuardar(Request $request)
    {
        if ($request->tipo_solicitud == 1) {
            $ingreso_almacen = new IngresoAlmacen();
            $ingreso_almacen->id_usuario =  Auth::id();
            $ingreso_almacen->id_cliente = $request->id_cliente;
            $ingreso_almacen->id_producto = $request->id_producto;
            $ingreso_almacen->cantidad = $request->cantidad;
            $ingreso_almacen->id_op = 999999;
            $ingreso_almacen->id_op_parcial = 999999;
            $ingreso_almacen->observaciones = $request->observaciones ?? '';
            $ingreso_almacen->status = 1;
            $ingreso_almacen->save();
        } else {
            $totalProduccion = $request->cantidad;
            $inventarios = IngresoAlmacen::where('status', 1)
                ->where('id_cliente', 999999)
                ->where('cantidad', '>', 0)
                ->where('id_producto', $request->id_producto)
                ->get();
            foreach ($inventarios as $inv) {
                if ($inv->cantidad >= $totalProduccion) {
                    if ($totalProduccion > 0) {
                        $rebajar = IngresoAlmacen::where('id', $inv->id)->first();
                        $rebajar->cantidad = floatval($rebajar->cantidad) - $totalProduccion;
                        $rebajar->observacion_rebaja = 'Cantidad=' . $totalProduccion . ' ' . $request->observaciones;
                        $rebajar->update();
                        $totalProduccion = 0;
                    }
                } else {
                    $rebajar = IngresoAlmacen::where('id', $inv->id)
                        ->first();
                    $totalProduccion = $totalProduccion - floatval($rebajar->cantidad);
                    $rebajar->cantidad = 0;
                    $rebajar->observacion_rebaja = 'Cantidad=' . $totalProduccion . ' ' . $request->observaciones;
                    $rebajar->update();
                }
            }
        }
        $arrResponse = [
            'message' => 'Inventario actualizado'
        ];
        return redirect()->route('inventario.dashboard_inventario')->with($arrResponse);
    }
    public function ajusteInventarioPrimaGuardar(Request $request)
    {
        if ($request->tipo_solicitud == 1) {
            $ingreso_almacen = new InventarioMateriaPrimas();
            $ingreso_almacen->id_usuario =  Auth::id();
            $ingreso_almacen->id_producto = $request->id_producto;
            $ingreso_almacen->cantidad = $request->cantidad;
            $ingreso_almacen->total = $request->total;
            $ingreso_almacen->observaciones = $request->observaciones ?? '';
            $ingreso_almacen->save();
            $arrResponse = [
                'message' => 'Inventario actualizado'
            ];
            return redirect()->route('inventario.dashboard_inventario_materia_prima')->with($arrResponse);
        } else {
            $total_quitar = floatval($request->total);
            $inventarios = InventarioMateriaPrimas::where('id_producto', $request->id_producto)
                ->where('total', '>', 0)
                ->get();

            foreach ($inventarios as $inv) {
                if ($inv->total >= $total_quitar) {
                    $rebajar = InventarioMateriaPrimas::where('id', $inv->id)
                        ->first();
                    $rebajar->total = floatval($rebajar->total) - $total_quitar;
                    $rebajar->update();
                    $total_quitar = 0;
                } else {
                    $rebajar = InventarioMateriaPrimas::where('id', $inv->id)
                        ->first();
                    $total_quitar = $total_quitar - floatval($rebajar->total);
                    $rebajar->total = 0;
                    $rebajar->update();
                }
            }
            if ($total_quitar == 0) {
                $ingreso_almacen = new InventarioMateriaPrimasSalida();
                $ingreso_almacen->id_usuario =  Auth::id();
                $ingreso_almacen->id_producto = $request->id_producto;
                $ingreso_almacen->cantidad = $request->cantidad;
                $ingreso_almacen->total = $request->total;
                $ingreso_almacen->observaciones = $request->observaciones ?? '';
                $ingreso_almacen->save();
            } else {
                $ingreso_almacen = new InventarioMateriaPrimasSalida();
                $ingreso_almacen->id_usuario =  Auth::id();
                $ingreso_almacen->id_producto = $request->id_producto;
                $ingreso_almacen->cantidad = $request->cantidad;
                $ingreso_almacen->total = floatval($request->total) - $total_quitar;
                $ingreso_almacen->observaciones = $request->observaciones ?? '';
                $ingreso_almacen->save();
            }

            $arrResponse = [
                'message' => 'Inventario actualizado'
            ];
            return redirect()->route('inventario.dashboard_inventario_materia_prima')->with($arrResponse);
        }
    }
}
